// Define the needed modules.
const fsp = require("fs/promises");
const path = require("path");
const { createCanvas, loadImage } = require("canvas");

// Define the location to our input.
const inputLocation = path.join(__dirname, "input");
const sourceLocation = path.join(__dirname, "src");

// Start the funky flow.
(async () => {
    // Grab the layout information.
    const layout = JSON.parse(await fsp.readFile(path.join(inputLocation, "layout.json"))).layout;

    // Create our canvas.
    const canvas = createCanvas(1920, 1080);
    const ctx = canvas.getContext("2d");

    // Load the emblem and logo images.
    const factionImage = await loadImage(path.join(sourceLocation, layout.faction + ".png"));
    const logoImage = await loadImage(path.join(sourceLocation, "title.png"));
    const backgroundImage = await loadImage(path.join(inputLocation, "image.png"));
    const vignetteImage = await loadImage(path.join(sourceLocation, "vignette.png"));

    // Draw the correct emblem on the canvas.
    ctx.drawImage(factionImage, 0, 0);

    // Define the highlight canvas that will allow us to draw out highlight and apply it after.
    const highlightCanvas = createCanvas(1920, 1080);
    const highlightCtx = highlightCanvas.getContext('2d');

    // Set the highlight properties.
    highlightCtx.shadowColor = '#' + layout.lighting.color + 'F2';
    highlightCtx.shadowBlur = 0;
    // Define the radius offsets.
    let radiusOffset = [
        (layout.lighting.direction == "right") ? 1 : -1,
        (layout.lighting.direction == "right") ? -1 : 1
    ];
    // Define the border radius.
    const borderRadius = 3;
    // Perform this operation to make the radius negative.
    highlightCtx.shadowOffsetX = borderRadius * radiusOffset[0];
    highlightCtx.shadowOffsetY = borderRadius * -1;
    // Draw the current canvas on the highlightCanvas.
    highlightCtx.drawImage(canvas, borderRadius * radiusOffset[1], borderRadius);

    // Get the image data
    const imageData = highlightCtx.getImageData(0, 0, highlightCanvas.width, highlightCanvas.height);
    for (let i = 0; i < imageData.data.length; i += 4) {
        // If the current image is completely opaque then make it transparent.
        if (imageData.data[i + 3] == 255) {
            imageData.data[i + 3] = 0;
        }
    }
    // Put the modified image data back on the canvas.
    highlightCtx.putImageData(imageData, 0, 0);

    // Set drop shadow properties
    const shadowDistance = 3;
    ctx.shadowColor = '#000000ff';
    ctx.shadowBlur = 10;
    ctx.shadowOffsetX = (layout.lighting.direction == "right") ? -shadowDistance : shadowDistance;
    ctx.shadowOffsetY = shadowDistance;

    // Draw the correct emblem on the canvas.
    ctx.drawImage(factionImage, 0, 0);

    // Reset the shadow.
    ctx.shadowColor = "#000000"


    // Draw the correct background.
    ctx.drawImage(backgroundImage, 0, 0);

    // Draw the vignette.
    ctx.drawImage(vignetteImage, 0, 0);

    // Draw the correct emblem on the canvas.
    ctx.drawImage(factionImage, 0, 0);

    // Set the blending mode to 'multiply'
    ctx.globalCompositeOperation = "normal";
    // Draw the highlighted area on the canvas.
    ctx.drawImage(highlightCanvas, 0, 0);
    // Reset blending mode.
    ctx.globalCompositeOperation = 'source-over';

    // Draw the title on the image.
    ctx.drawImage(logoImage, 0, 0);

    // Output the image.
    await fsp.writeFile(path.join(__dirname, "output.png"), canvas.toBuffer(), "binary");
})();